#!/bin/bash

check_dependency() {
  if ! command -v "$1" > /dev/null 2>&1; then
    echo "Error: '$1' is not installed. Please install it using 'pkg install $1'."
    exit 1
  fi
}

# Check for jq, unzip, and curl
check_dependency jq
check_dependency unzip
check_dependency curl

CONFIG_FILE="patch-config.txt"

# Check if CONFIG_FILE exists
if [ ! -f "$CONFIG_FILE" ]; then
    echo "Config file '$CONFIG_FILE' not found! Assuming no patching needed."
    exit 0
fi

# Convert line endings to Unix format
sed -i 's/\r$//' "$CONFIG_FILE"

# Debug information
echo "Current directory: $(pwd)"
echo "Config file path: $(pwd)/$CONFIG_FILE"

# Read configuration from file
. "$(pwd)/$CONFIG_FILE"

# Get the latest hash
echo "Getting latest commit SHA hash"
latest_patch_sha=$(curl -s "https://gitgud.io/api/v4/projects/$username%2F$repo/repository/branches/$branch" | jq -r '.commit.id')

download_extract() {
    # Download zip file
    echo "Downloading latest patch..."
    curl -sL "https://gitgud.io/$username/$repo/-/archive/$branch/$repo-$branch.zip" -o repo.zip

    # Extract contents, overwriting conflicts
    echo "Extracting..."
    unzip -qo repo.zip

    echo "Applying patch..."
    cp -r "$repo-$branch/"* .

    echo "Cleaning up..."
    rm repo.zip
    rm -rf "$repo-$branch"
    rm -f latest_patch_sha.txt

    # Store latest SHA for next check
    echo "$latest_patch_sha" > previous_patch_sha.txt
}

# Check if previous_patch_sha.txt exists
if [ ! -f previous_patch_sha.txt ]; then
    echo "Previous SHA hash not found!"
    echo "Assuming first time patching..."
    download_extract
else
    # Read the stored SHA from previous check
    previous_patch_sha=$(cat previous_patch_sha.txt)

    # Compare trimmed SHAs
    if [ "$latest_patch_sha" != "$previous_patch_sha" ]; then
        echo "Update found! Patching..."
        download_extract
    else
        echo "Patch is up to date."
    fi
fi