@echo off
setlocal EnableExtensions EnableDelayedExpansion

echo Checking for pwsh...
set _my_shell=pwsh

REM Check if pwsh.exe exists
where /q !_my_shell!
if !errorlevel! neq 0 (
  echo pwsh not found. Falling back to powershell...
  REM If pwsh is not found, set to powershell
  set _my_shell=powershell

  REM Check if powershell.exe exists
  echo Checking for powershell...
  where /q !_my_shell!
  if !errorlevel! neq 0 (
    echo.Error: Powershell not found!
    pause
    exit /B 1
  ) else (
    echo powershell found.
  )
) else (
  echo pwsh found.
)

echo Using !_my_shell! for script execution.

REM Check if patch-config.txt exists
if not exist patch-config.txt (
    echo "Config file (patch-config.txt) not found! Assuming no patching needed."
    pause
    exit /b
)

REM Read configuration from file
for /f "tokens=1,2 delims==" %%a in (patch-config.txt) do (
    if "%%a"=="username" set "username=%%b"
    if "%%a"=="repo" set "repo=%%b"
    if "%%a"=="branch" set "branch=%%b"
)

REM Get the latest hash
echo "Getting latest commit SHA hash"
!_my_shell! -Command "(Invoke-RestMethod -Uri 'https://gitgud.io/api/v4/projects/%username%%%2F%repo%/repository/branches/%branch%').commit.id" > latest_patch_sha.txt

REM Read the latest SHA from the file
set /p latest_patch_sha=<latest_patch_sha.txt

REM Check if previous_patch_sha.txt exists
if not exist previous_patch_sha.txt (
    echo "Previous SHA hash not found!"
    echo "Assuming first time patching..."
    goto download_extract
)

REM Read the stored SHA from previous check
set /p previous_patch_sha=<previous_patch_sha.txt

REM Trim whitespace from SHA strings
set "previous_patch_sha=%previous_patch_sha: =%"
set "latest_patch_sha=%latest_patch_sha: =%"

REM Compare trimmed SHAs
if "%latest_patch_sha%" neq "%previous_patch_sha%" (
    echo "Update found! Patching..."
    goto download_extract
) else (
    echo "Patch is up to date."
)

REM Delete latest_patch_sha.txt
del latest_patch_sha.txt

endlocal
pause
exit /b

:download_extract

REM Escape single quotes in paths
set "escaped_cd=%CD:'=''%" 

REM Download zip file
echo "Downloading latest patch..."
!_my_shell! -Command "Set-Location -LiteralPath '%escaped_cd%'; Invoke-WebRequest -Uri 'https://gitgud.io/%username%/%repo%/-/archive/%branch%/%repo%-%branch%.zip' -OutFile 'repo.zip'"
if !errorlevel! neq 0 (
    pause
    exit /b
)

REM Extract contents, overwriting conflicts
echo "Extracting..."
!_my_shell! -Command "Set-Location -LiteralPath '%escaped_cd%'; Expand-Archive -Path '.\repo.zip' -DestinationPath '.' -Force"
echo "Applying patch..."
xcopy /s /e /y "%repo%-%branch%\*" "."

REM Clean up
echo "Cleaning up..."
del repo.zip
rmdir /s /q "%repo%-%branch%"
del latest_patch_sha.txt

REM Store latest SHA for next check
echo %latest_patch_sha% > previous_patch_sha.txt
endlocal
pause
exit /b
